<?php

namespace Core;

class Request {

    // Sanitize input to prevent SQL injection and XSS
    public static function sanitize($input, $type = 'string') {
        // Handle different types of input sanitization
        switch ($type) {
            case 'string':
                return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
            case 'int':
                return filter_var($input, FILTER_SANITIZE_NUMBER_INT);
            case 'email':
                return filter_var($input, FILTER_SANITIZE_EMAIL);
            case 'url':
                return filter_var($input, FILTER_SANITIZE_URL);
            case 'text':
                return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
            default:
                return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
        }
    }

    // Validate the input based on type
    public static function validate($input, $type = 'string') {
        // Trim the input to remove any unnecessary whitespace
        $input = trim($input);
    
        switch ($type) {
            case 'string':
                // Ensure it's not empty and contains only alphabets or digits
                return !empty($input); // Adjust this if you need to validate specific characters
    
            case 'int':
                // Validate that the input is a valid integer
                return filter_var($input, FILTER_VALIDATE_INT) !== false;
    
            case 'email':
                // Validate that the input is a valid email format
                return filter_var($input, FILTER_VALIDATE_EMAIL) !== false;
    
            case 'url':
                // Validate that the input is a valid URL
                return filter_var($input, FILTER_VALIDATE_URL) !== false;
    
            case 'text':
                // Ensure input is not empty for a general text field
                return !empty($input);
    
            case 'float':
                // Validate that the input is a valid float
                return filter_var($input, FILTER_VALIDATE_FLOAT) !== false;
    
            case 'phone':
                // Validate that the input is a valid phone number (adjust regex for different countries)
                return preg_match("/^\+?[0-9]{7,15}$/", $input);
    
            case 'alphanumeric':
                // Validate that the input contains only letters and numbers
                return preg_match("/^[a-zA-Z0-9]+$/", $input);
    
            case 'datetime':
                // Validate that the input is a valid datetime (in Y-m-d H:i:s format)
                return preg_match("/\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}/", $input);
    
            default:
                // Default to checking if input is not empty
                return !empty($input);
        }
    }
    

    // Method to retrieve and sanitize POST data
    public static function getPost($key, $type = 'string') {
        return isset($_POST[$key]) ? self::sanitize($_POST[$key], $type) : null;
    }

    // Method to retrieve and sanitize GET data
    public static function getGet($key, $type = 'string') {
        return isset($_GET[$key]) ? self::sanitize($_GET[$key], $type) : null;
    }

    // Method to retrieve and sanitize input from other sources like files
    public static function getFile($key) {
        return isset($_FILES[$key]) ? $_FILES[$key] : null;
    }

    // Method to handle other kinds of request data
    public static function get($key, $source = 'POST', $type = 'string') {
        $input = null;
        switch ($source) {
            case 'POST':
                $input = self::getPost($key, $type);
                break;
            case 'GET':
                $input = self::getGet($key, $type);
                break;
            default:
                // Can extend for other sources like files, cookies, etc.
                break;
        }
        return $input;
    }
}
