<?php

// Top widget now uses local customer usage database for online/offline counts
// This eliminates RouterOS API calls and provides faster, more reliable data

class top_widget
{
    public function getWidget()
    {
        global $ui, $current_date, $start_date;

        $iday = ORM::for_table('tbl_transactions')
            ->where('recharged_on', $current_date)
            ->where_not_equal('method', 'Customer - Balance')
            ->where_not_equal('method', 'Recharge Balance - Administrator')
            ->where_not_equal('method', 'Voucher') // Exclude voucher
            
            ->sum('price');

        if ($iday == '') {
            $iday = '0.00';
        }
        $ui->assign('iday', $iday);

        $imonth = ORM::for_table('tbl_transactions')
            ->where_not_equal('method', 'Customer - Balance')
            ->where_not_equal('method', 'Recharge Balance - Administrator')
            ->where_not_equal('method', 'Voucher') // Exclude voucher
            ->where_gte('recharged_on', $start_date)
            ->where_lte('recharged_on', $current_date)->sum('price');
        if ($imonth == '') {
            $imonth = '0.00';
        }
        $ui->assign('imonth', $imonth);

        // Get online users from local customer usage database (fastest and most reliable)
        try {
            $online_data = $this->getOnlineUsersFromLocal(true); // pass true to get lists
            $u_act = $online_data['online'];
            $u_all = $online_data['offline'];
            $active_but_not_online = $online_data['active_but_not_online'] ?? [];
        } catch (Exception $e) {
            $u_act = ORM::for_table('tbl_user_recharges')->where('status', 'on')->count();
            $total_active_accounts = $u_act;
            $u_all = 0;
            $active_but_not_online = [];
            $online_data = [
                'online' => $u_act,
                'offline' => $u_all,
                'total_tracked' => 0,
                'cached' => false,
                'timestamp' => time(),
                'status' => 'fallback',
                'source' => 'Active Accounts Fallback',
                'active_but_not_online' => []
            ];
        }
        $ui->assign('u_act', $u_act);
        $ui->assign('u_all', $u_all);
        $ui->assign('active_but_not_online', $active_but_not_online);
        
        // Add status information for display
        $ui->assign('online_cached', $online_data['cached'] ?? false);
        $ui->assign('online_source', $online_data['source'] ?? 'Local Database');
        $ui->assign('online_status', $online_data['status'] ?? 'live');
        $ui->assign('last_update', $online_data['timestamp'] ?? time());
        $ui->assign('total_tracked', $online_data['total_tracked'] ?? 0);
        
        // Add data source indicator for admins
        $data_source_class = '';
        switch($online_data['source'] ?? '') {
            case 'Local Usage Database':
                $data_source_class = 'success'; // Green - most reliable
                break;
            case 'Active Accounts Fallback':
                $data_source_class = 'secondary'; // Gray - fallback
                break;
            default:
                $data_source_class = 'warning'; // Yellow - error or unknown
        }
        $ui->assign('data_source_class', $data_source_class);


        $c_all = ORM::for_table('tbl_customers')->count();
        if (empty($c_all)) {
            $c_all = '0';
        }
        $ui->assign('c_all', $c_all);
        return $ui->fetch('widget/top_widget.tpl');
    }

    /**
     * Get online/offline users from local customer usage database
     * This uses tbl_usage_sessions and tbl_user_recharges tables for accurate counts
     */
    private function getOnlineUsersFromLocal($withList = false)
    {
        try {
            // Define threshold for "online" status (last 1 minute - matches system-wide setting)
            $threshold = date('Y-m-d H:i:s', strtotime('-1 minute'));
            
            // Get total active accounts (have active packages)
            $total_active_accounts = ORM::for_table('tbl_user_recharges')
                ->where('status', 'on')
                ->count();
            
            // Get online users from usage sessions table
            $online_count = 0;
            $active_but_not_online = [];
            
            try {
                // Check if usage tables exist
                $table_exists = ORM::for_table('tbl_usage_sessions')->limit(1)->find_one();
                
                if ($table_exists !== false) {
                    // Get online users (seen within threshold)
                    $online_users = ORM::for_table('tbl_usage_sessions')
                        ->select('username')
                        ->where_gte('last_seen', $threshold)
                        ->group_by('username')
                        ->find_array();
                    
                    $online_count = count($online_users);
                    
                    if ($withList) {
                        // Get active accounts that are not online
                        $online_usernames = array_column($online_users, 'username');
                        
                        $active_accounts = ORM::for_table('tbl_user_recharges')
                            ->select('username')
                            ->where('status', 'on')
                            ->find_array();
                        
                        foreach ($active_accounts as $account) {
                            if (!in_array($account['username'], $online_usernames)) {
                                $active_but_not_online[] = $account['username'];
                            }
                        }
                    }
                } else {
                    // Table doesn't exist yet, throw exception
                    throw new Exception("Usage sessions table (tbl_usage_sessions) does not exist");
                }
            } catch (Exception $e) {
                // Usage tables don't exist or error, throw exception
                throw new Exception("Usage tables not available: " . $e->getMessage());
            }
            
            // Calculate inactive users (active accounts that are not online)
            $inactive_count = max(0, $total_active_accounts - $online_count);
            
            return [
                'online' => $online_count,
                'offline' => $inactive_count,
                'total_tracked' => $total_active_accounts,
                'total_active_accounts' => $total_active_accounts,
                'cached' => false,
                'timestamp' => time(),
                'status' => 'live',
                'source' => 'Local Usage Database',
                'threshold_minutes' => 2,
                'active_but_not_online' => $active_but_not_online
            ];
            
        } catch (Exception $e) {
            // No fallback - return error state
            return [
                'online' => 0,
                'offline' => 0,
                'total_tracked' => 0,
                'total_active_accounts' => 0,
                'cached' => false,
                'timestamp' => time(),
                'status' => 'error',
                'source' => 'Database Error: ' . $e->getMessage(),
                'threshold_minutes' => 2,
                'active_but_not_online' => []
            ];
        }
    }

    
    /**
     * Get fresh online users data for AJAX requests
     * Uses the same local database method for consistency
     */
    public function getOnlineUsersAjax()
    {
        return $this->getOnlineUsersFromLocal();
    }
    
    /**
     * Static method to get fresh data for AJAX endpoint
     */
    public static function ajaxGetOnlineUsers()
    {
        $widget = new self();
        return $widget->getOnlineUsersAjax();
    }
}
